-- File is True Dual Port RAM with dual clock for DRAM and ROM
-- Copyright (C) 2019  Dr. Uwe Meyer-Baese.
-- ===================================================================
LIBRARY ieee;
USE ieee.STD_LOGIC_1164.ALL;
USE ieee.STD_LOGIC_arith.ALL;
USE ieee.STD_LOGIC_unsigned.ALL;
-- ===================================================================
ENTITY dpram4Kx32 IS
PORT (clk_a  : IN STD_LOGIC; -- System clock DRAM
      clk_b  : IN STD_LOGIC; -- System clock PROM
      addr_a : IN STD_LOGIC_VECTOR(11 DOWNTO 0); -- Data mem. address
      addr_b : IN STD_LOGIC_VECTOR(11 DOWNTO 0); -- Prog. mem. address
      data_a : IN STD_LOGIC_VECTOR(31 DOWNTO 0); -- Data in for DRAM 
      we_a   : IN STD_LOGIC := '0'; -- Write only DRAM
      q_a    : OUT STD_LOGIC_VECTOR(31 DOWNTO 0); -- DRAM output
      q_b    : OUT STD_LOGIC_VECTOR(31 DOWNTO 0)); -- ROM output
END ENTITY dpram4Kx32;
-- ===================================================================
ARCHITECTURE fpga OF dpram4Kx32 IS

-- Build a 2-D array type for the RAM
TYPE MEM IS ARRAY(0 TO 4095) of STD_LOGIC_VECTOR(31 DOWNTO 0);
 
-- Define RAM and initial values
SHARED VARIABLE dram : MEM  := (
-- This style would require a PC relative addressing
X"E51F_D000", --        ldr     sp, [pc, #0]    /* initial stack pointer */
X"EA00_0000", --        b  main    /* initial global pointer */
X"3FFF_FFEC", --        .word 0x3fffffec     /* constant */       
--X"E30F_DFEC", --       movw    sp, #0xfffc      /* LSBs sp */
--X"E343_DFFF", --       movt    sp, #0x3fff      /* MSBs sp */
X"E24D_D040", -- main:  sub sp, sp, #64      /* make space on stack */
-- =============- s = 1 =============
X"E3A0_3001", --        mov r3, #1      /* s= 1 */
X"E58D_303C", --        str  r3, [sp, #60]  /* store s */
-- =============- g = 2 =============
X"E300_3BA4", --       movw    r3, #2980      /* pointer g */
X"E340_3000", --       movt    r3, #0         /* MSBs g */
X"E3A0_2002", --       mov     r2, #2         /* LSBS */
X"E583_2000", --       str     r2, [r3]       /* store g */
-- =============- sarray[0] = s =============
X"E59D_203C", --       ldr     r2, [sp, #60]  /* address sarray[0] */
X"E58D_2004", --       str     r2, [sp, #4]   /* sarray[0]=s */
-- =============- garray[0] = g =============
X"E593_2000", --       ldr     r2, [r3]        /* address garray[0] */
X"E583_2004", --       str     r2, [r3, #4]    /* store garray[0]=g */
-- =============- sarray[s] = 0x1357 =============
X"E59D_203C", --       ldr     r2, [sp, #60]   /* s */
X"E28D_1040", --       add     r1, sp, #64     /* mv sp to r1*/
X"E081_2102", --       add     r2, r1, r2, lsl #2 /* s*4+sp = &sarray[s]*/
X"E301_1357", --       movw    r1, #4951       /* =0x1357  */
X"E502_103C", --       str     r1, [r2, #-60]  /* store sarray[s] */
-- =============- garray[g] = 0x2468 =============
X"E5932000", --       ldr     r2, [r3]         /* load g */
X"E0833102", --       add     r3, r3, r2, lsl #2 /* g*4+&garray[0] */
X"E3022468", --       movw    r2, #9320       /* =0x2468 */
X"E5832004", --       str     r2, [r3, #4]    /* store garray[g] */
X"EAFFFFEB", --        b     main             /* start loop again */
--OTHERS => X"E320F000"); -- default NOP
OTHERS => "UUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU"); -- Default unknown mem

BEGIN

  -- Port A aka DRAM
  PROCESS(clk_a)
  BEGIN
    IF(RISING_EDGE(clk_a)) THEN 
      IF(we_a = '1') THEN
        dram(CONV_INTEGER('0'& addr_a)) := data_a;
        q_a <= data_a;
      ELSE
        q_a <= dram(CONV_INTEGER('0'& addr_a));
      END IF;
    END IF;
  END PROCESS;

  -- Port B aka ROM
  PROCESS(clk_b)
  BEGIN
    IF(RISING_EDGE(clk_b)) THEN 
      q_b <= dram(CONV_INTEGER('0' & addr_b));
    END IF;
  END PROCESS;
  
END fpga;
